/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

/*
  find possible deadlocks in a routes file.
  finucane@myri.com
*/


#include <stdarg.h>

#include "lx.h"
#include "lx_map_file.h"
#include "lx_routing.h"
#include "lx_route_file.h"

lx_t lx;

static void usage (char*s)
{
  fprintf (stderr, "usage: %s <map file> [<route file>]\n", s);
  exit (0);
}


static int find_cycle (lx_map_t*m, lx_node_t*n, int p, int level)
{
  int i, last_port;
  lx_node_t*to;
  
  insist (m);
  
  if (!n || !n->xbar)
    return 0;

  /*
  for (i = 0; i < level; i++)
    printf (" ");

  printf (lx_node_format ":%d\n", lx_node_args (n), p);
  */
  if (n->node_data.seen [p] == 1)
  {
    printf ("deadlock detected. printing loop backwards\n");
    printf (lx_node_format ":%d\n", lx_node_args (n), p);
    return 1;
  }
	
  n->node_data.seen [p] = 1;

  for (i = 0; i < LX_XBAR_SIZE; i++)
  {
    if (n->node_data.hops [p][i].node)
    {
      insist (lx_get_node (n, i) == n->node_data.hops [p][i].node);
      
      if (find_cycle (m, n->node_data.hops [p][i].node, lx_get_port (n, i), level + 1))
      {
	to = lx_follow (n->node_data.hops [p][i].root, n->node_data.hops [p][i].iport, n->node_data.hops [p][i].route, &last_port);
	insist (to);
	
	printf (lx_node_format ":(%dx%d) "
		lx_node_format " %s ",
		lx_node_args (n), p, i,
		lx_node_args (n->node_data.hops [p][i].root), 
		lx_print_route (n->node_data.hops [p][i].route->hops, n->node_data.hops [p][i].route->length));
	mi_print_hops (stdout, n->node_data.hops [p][i].root, to, n->node_data.hops [p][i].iport, last_port, n->node_data.hops [p][i].route);
	
	n->node_data.seen [p] = 0;
	return 1;
      }
    }
  }
  n->node_data.seen [p] = 0;

  except: return 0;
}

static int add_route (lx_map_t*m, lx_node_t*n, int p, lx_route_t*r)
{
  int i, p1;
  lx_node_t*h = n;
  lx_node_t*n1;
  int added_something = 0;
  int iport = p;
  
  insist (m && n && !n->xbar && r);  
  insist (p >= 0 && p < LX_HOST_SIZE);
  
  for (i = -1; i < r->length; i++)
  {
    int o = p + (i >= 0 ? r->hops [i] : 0);
    p1 = lx_get_port (n, o);
    n1 = lx_get_node (n, o);
    insist (n1);
  
    if (n->xbar && !n->node_data.hops [p][o].node)
    {
      added_something = 1;
      
      n->node_data.hops [p][o].node = n1;
      n->node_data.hops [p][o].root = h;
      n->node_data.hops [p][o].route = r;
      n->node_data.hops [p][o].iport = iport;
    }
    
    n = n1;
    p = p1;
  }
  except: return added_something;
}

int main (int argc, char*argv [])
{
  lx_map_t*m;
  FILE*fp;
  int i, j, o, k, from, num_hosts, num_ports, num_routes, line_count;
  lx_route_table_entry_t*entries, *e;
  lx_node_t*first;
  lx_node_t*h;
  char s [100];
  
  mi_verbose_flag = 0;

  srand (time (0));
  
  if (argc < 2)
    usage (argv [0]);
  
  if (!(fp = fopen (argv [1], "r")))
  {
    fprintf (stderr, "couldn't open %s\n", argv [1]);
    usage (argv [0]);
  }
  
  lx_init (&lx);
  
  if (!lx_map_file_read (fp, &lx.maps [0], &first))
  {
    fprintf (stderr, "lx_map_file_read () failed\n");
    return 0;
  }

  m = lx.maps;
  fprintf (stderr, "%d hosts and %d xbars\n", m->num_hosts, m->num_xbars);

  if (!(fp = argc == 2 ? stdin : fopen (argv [2], "r")))
  {
    fprintf (stderr, "couldn't open %s", argv [2]);
    usage (argv [0]);
  }
  
  line_count = 0;

  *s = 0;
  
  while ((entries = lx_route_file_read_routes (fp, &from, &num_hosts, &num_ports, &num_routes, &line_count)))
  {
    insist (from >= 0 && from < m->num_hosts);
    insist (num_hosts == m->num_hosts);
    insist (num_ports > 0 && num_ports <= LX_HOST_SIZE);
    insist (num_routes > 0);
   
    h = m->host_array [from];
    insist (h && !h->xbar);
    
    for (i = 0; i < num_hosts; i++)
    {
      for (j = 0; j < num_ports; j++)
      {
	for (o = 0; o < num_ports; o++)
	{ 
	  for (k = 0; k < num_routes; k++)
	  {
	    e = lx_route_file_entry (entries, num_ports, num_hosts, num_routes, j, i, o, k);
	    insist (e);

	    if (e->valid)
	      add_route (m, h, j, &e->route);
	  }
	}
      }
    }
  }

  for (i = 0; i < num_hosts; i++)
  {
    for (j = 0; j < LX_HOST_SIZE; j++)
    {
      //printf ("host %d\n", i);
      if (find_cycle (m, lx_get_node (m->host_array [i], j), lx_get_port (m->host_array [i], j), 0))
	return 0;
    }
  }

  free (entries);
  printf ("cannot deadlock\n");
  return 1;
  except: return 0;
}
